using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using Newtonsoft.Json;

namespace POIApp
{
	public class POIJsonService : IPOIDataService
	{
		private string _storagePath;
		private List<PointOfInterest> _pois = new List<PointOfInterest>();

		public POIJsonService (string storagePath)
		{
			_storagePath = storagePath;

			// create the storage path if it does not exist
			if (!Directory.Exists(_storagePath))
				Directory.CreateDirectory(_storagePath);

			RefreshCache ();
		}
		
		#region IPOIService implementation

		public PointOfInterest GetPOI (int id)
		{
			PointOfInterest poi = _pois.Find (p => p.Id == id);
			return poi;
		}

		private int GetNextId()
		{
			if (_pois.Count == 0)
				return 1;
			else
				return _pois.Max (p => p.Id.Value) + 1;
		}

		private string GetFilename(int id)
		{
			return Path.Combine (_storagePath, "poi" + id.ToString () + ".json");
		}

		public void SavePOI (PointOfInterest poi)
		{
			Boolean newPOI = false;
			if (!poi.Id.HasValue) {
				poi.Id = GetNextId ();
				newPOI = true;
			}

			string poiString = JsonConvert.SerializeObject (poi);
			File.WriteAllText (GetFilename (poi.Id.Value), poiString);

			if (newPOI)
				_pois.Add (poi);
		}

		public void DeletePOI (PointOfInterest poi)
		{
			File.Delete (GetFilename (poi.Id.Value));
			_pois.Remove (poi);
		}

		public IReadOnlyList<PointOfInterest> POIs {
			get { return _pois; }
		}

		public void RefreshCache()
		{
			_pois.Clear ();

			string[] filenames = Directory.GetFiles (_storagePath, "*.json");
			foreach (string filename in filenames) {
				string poiString = File.ReadAllText (filename);
				PointOfInterest poi = JsonConvert.DeserializeObject<PointOfInterest> (poiString);
				_pois.Add (poi);
			}
		}

		#endregion
	}
}

